<?php if ( ! defined('BASEPATH')) exit('No direct script access allowed');

/**
* @package direct-project-innovation-initiative
* @subpackage core
*/

/** */
require_once APPPATH.'helpers/environment_helper.php'; //this class is loaded before the loader class, can't rely on loading dependencies normally
require_once APPPATH.'helpers/debug_helper.php';

/**
* DPII CodeIgniter Security Class Extension
*
* @package direct-project-innovation-initiative
* @subpackage core
* @author		         
*/
class DPII_Security extends CI_Security {
	public function __construct()
	{
		parent::__construct();
		//check for mcrypt, necessary to encrypt the CSRF cookie contents
		$mcrypt_exists = ( ! function_exists('mcrypt_encrypt')) ? FALSE : TRUE;
		if(!$mcrypt_exists) {
			show_error('Mcrypt extension must be loaded in order to use the extended Security class.');
		}
		//check for encryption key to perform cookie encryption
		if(config_item('encryption_key') == '') {
			show_error('In order to use the extended Security class, it is required that you set an encryption key in your config file.');
		}
	}
	
	/**
	 * Verify Cross Site Request Forgery Protection
	 *
	 * @return	object
	 */
	public function csrf_verify()
	{
		global $RTR; //this method gets run early enough that we don't have access to the controller or the standard router - grab the early global version of the router class instead
		
		//VLER addition - require GET calls via AJAX to contain the CSRF token in their URLS
		if(IS_AJAX && strtoupper($_SERVER['REQUEST_METHOD']) === 'GET' && $RTR->fetch_class() != 'auth' && $RTR->fetch_method() != 'logout'){
			
			//make sure we have a token in GET
			if(empty($_GET[$this->_csrf_token_name])){
				return $this->csrf_show_error('the CSRF token name ('.$this->_csrf_token_name.') is not in $_GET ');
			}
			
			//make sure we have a token in COOKIE
			if(empty($_COOKIE[$this->_csrf_cookie_name])){
				return $this->csrf_show_error('the CSRF cookie name ('.$this->_csrf_cookie_name.') is not in $_COOKIE ');
			}
		
			//make sure the COOKIE and GET tokens match
			$cookie_token = $this->decode(base64_decode($_COOKIE[$this->_csrf_cookie_name]));
			if($_GET[$this->_csrf_token_name] != $cookie_token){
				return $this->csrf_show_error('the GET token and the COOKIE token do not match. ('.$_GET[$this->_csrf_token_name].' != '.$cookie_token.')');
			}
			
			return $this; //don't reset the token after verifying - async calls will probably be followed with more calls relying on the same token
		}
		
		
		// If it's not a POST request we will set the CSRF cookie
		if (strtoupper($_SERVER['REQUEST_METHOD']) !== 'POST')
		{	
			return $this->csrf_set_cookie();
		}

		// Do the tokens exist in both the _POS   n  _COOKIE arrays?
		if ( ! isset($_POST[$this->_csrf_token_name], $_COOKIE[$this->_csrf_cookie_name]))
		{
			$this->csrf_show_error();
		}

		// Do the tokens match?
		if ($_POST[$this->_csrf_token_name] != $this->decode(base64_decode($_COOKIE[$this->_csrf_cookie_name])))
		{
			$this->csrf_show_error();
		}

		// We kill this since we're done and we don't want to
		// pollute the _POST array
		unset($_POST[$this->_csrf_token_name]);

		// Nothing should last forever
		unset($_COOKIE[$this->_csrf_cookie_name]);
		$this->_csrf_set_hash();
		$this->csrf_set_cookie();

		log_message('debug', 'CSRF token verified');

		return $this;
	}
	
	/**
	 * Set Cross Site Request Forgery Protection Cookie
	 *
	 * @return	object
	 */
	public function csrf_set_cookie()
	{				
		$expire = time() + $this->_csrf_expire;
		$secure_cookie = (config_item('cookie_secure') === TRUE) ? 1 : 0;
		$httponly_cookie = (config_item('cookie_httponly') === TRUE) ? 1 : 0;
				
		//we are behind nginx, so check the proxy_set_header value of HTTP_HTTPS to determine if we are using HTTPS
		if ($secure_cookie && (empty($_SERVER['HTTP_HTTPS']) OR strtolower($_SERVER['HTTP_HTTPS']) === 'off'))
		{
			return FALSE;
		}
		
		setcookie($this->_csrf_cookie_name, base64_encode($this->encode($this->_csrf_hash)), $expire, config_item('cookie_path'), config_item('cookie_domain'), $secure_cookie, $httponly_cookie);
		log_message('debug', "CRSF cookie Set");

		return $this;
	}	
	
	public function csrf_show_error($reason_why_action_is_not_allowed = ''){
		global $RTR; //this method gets run early enough that we don't have access to the controller or the standard router - grab the early global version of the router class instead
		$controller = $RTR->fetch_class();
		$action = $RTR->fetch_method();
				
		//log the message instead of displaying an error so we won't interfere w/redirect		
		$message = 'The action you have requested ('.$controller.'/'.$action.') is not allowed';
		if(!empty($reason_why_action_is_not_allowed))
			$message .= ' because '.$reason_why_action_is_not_allowed;
		log_message('error', $message); 
		
		if($controller != 'auth' && $action != 'logout'){	
			header("Location: https://".DIRECT_DOMAIN.'/auth/logout', TRUE, 302); //redirect to the logout page
		}
		
		show_error('The action you have requested is not allowed', 401); //worst case scenario, our redirect didn't work - give the standard message to the user saying action is not allowed.
	}	
		
	/** Overrides parent to use the Codeigniter 2.2 version of this function.  Can be removed when we upgrade to 2.2 or beyond. */
	/*
	 * Remove Evil HTML Attributes (like evenhandlers and style)
	 *
	 * It removes the evil attribute and either:
	 * 	- Everything up until a space
	 *		For example, everything between the pipes:
	 *		<a |style=document.write('hello');alert('world');| class=link>
	 * 	- Everything inside the quotes
	 *		For example, everything between the pipes:
	 *		<a |style="document.write('hello'); alert('world');"| class="link">
	 *
	 * @param string $str The string to check
	 * @param boolean $is_image TRUE if this is an image
	 * @return string The string with the evil attributes removed
	 */
	protected function _remove_evil_attributes($str, $is_image)
	{
		// All javascript event handlers (e.g. onload, onclick, onmouseover), style, and xmlns
		$evil_attributes = array('on\w*', 'style', 'xmlns', 'formaction');

		if ($is_image === TRUE)
		{
			/*
			 * Adobe Photoshop puts XML metadata into JFIF images, 
			 * including namespacing, so we have to allow this for images.
			 */
			unset($evil_attributes[array_search('xmlns', $evil_attributes)]);
		}

		do {
			$count = 0;
			$attribs = array();

			// find occurrences of illegal attribute strings with quotes (042 and 047 are octal quotes)
			preg_match_all('/('.implode('|', $evil_attributes).')\s*=\s*(\042|\047)([^\\2]*?)(\\2)/is', $str, $matches, PREG_SET_ORDER);

			foreach ($matches as $attr)
			{
				$attribs[] = preg_quote($attr[0], '/');
			}

			// find occurrences of illegal attribute strings without quotes
			preg_match_all('/('.implode('|', $evil_attributes).')\s*=\s*([^\s>]*)/is', $str, $matches, PREG_SET_ORDER);

			foreach ($matches as $attr)
			{
				$attribs[] = preg_quote($attr[0], '/');
			}

			// replace illegal attribute strings that are inside an html tag
			if (count($attribs) > 0)
			{
				$str = preg_replace('/(<?)(\/?[^><]+?)([^A-Za-z<>\-])(.*?)('.implode('|', $attribs).')(.*?)([\s><]?)([><]*)/i', '$1$2 $4$6$7$8', $str, -1, $count);
			}

		} while ($count);

		return $str;
	}
	
	/**
	 * Set Cross Site Request Forgery Protection Cookie
	 *
	 * @return	string
	 */
	protected function _csrf_set_hash()
	{
		if ($this->_csrf_hash == '')
		{
			// If the cookie exists we will use its value.
			// We don't necessarily want to regenerate it with
			// each page load since a page could contain embedded
			// sub-pages causing this feature to fail
			if (isset($_COOKIE[$this->_csrf_cookie_name]) &&
				preg_match('#^[0-9a-f]{64}$#iS', $this->decode(base64_decode($_COOKIE[$this->_csrf_cookie_name]))) === 1)
			{
				return $this->_csrf_hash =  $this->decode(base64_decode($_COOKIE[$this->_csrf_cookie_name]));
			}
			return $this->_csrf_hash = hash('sha256',openssl_random_pseudo_bytes(32));
		}
		return $this->_csrf_hash;
	}

	/**
	 * Custom Mcrypt AES decryption for CSRF cookie decryption
	 *
	 * @param	string
	 * @return	string
	 */
	private function decode($data) {
		$key = hash('sha256',('encryption_key'),true);
		$init_size = mcrypt_get_iv_size(MCRYPT_RIJNDAEL_256, MCRYPT_MODE_CBC);
		
		if ($init_size > strlen($data))
		{
			return FALSE;
		}
		
		$init_vect = substr($data, 0, $init_size);
		$data = substr($data, $init_size);
		return mcrypt_decrypt(MCRYPT_RIJNDAEL_256, $key, $data, MCRYPT_MODE_CBC, $init_vect);
	}
	
	/**
	 * Custom Mcrypt AES encryption for CSRF cookie encryption
	 *
	 * @param	string
	 * @return	string
	 */
	private function encode($data) {
		$key = hash('sha256',('encryption_key'),true);
		$init_size = mcrypt_get_iv_size(MCRYPT_RIJNDAEL_256, MCRYPT_MODE_CBC);
		$init_vect = mcrypt_create_iv($init_size, MCRYPT_RAND);
		return $init_vect.mcrypt_encrypt(MCRYPT_RIJNDAEL_256, $key, $data, MCRYPT_MODE_CBC, $init_vect);
	}	
}